/*
 * Decompiled with CFR 0.152.
 */
package net.sf.samtools.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import net.sf.samtools.FileTruncatedException;
import net.sf.samtools.util.BlockCompressedFilePointerUtil;
import net.sf.samtools.util.BlockCompressedStreamConstants;
import net.sf.samtools.util.BlockGunzipper;
import net.sf.samtools.util.IOUtil;
import net.sf.samtools.util.SeekableBufferedStream;
import net.sf.samtools.util.SeekableFileStream;
import net.sf.samtools.util.SeekableHTTPStream;
import net.sf.samtools.util.SeekableStream;

public class BlockCompressedInputStream
extends InputStream {
    private InputStream mStream = null;
    private SeekableStream mFile = null;
    private byte[] mFileBuffer = null;
    private byte[] mCurrentBlock = null;
    private int mCurrentOffset = 0;
    private long mBlockAddress = 0L;
    private int mLastBlockLength = 0;
    private final BlockGunzipper blockGunzipper = new BlockGunzipper();
    private volatile ByteArrayOutputStream buf = null;
    private static final byte eol = 10;
    private static final byte eolCr = 13;

    public BlockCompressedInputStream(InputStream stream) {
        this.mStream = IOUtil.toBufferedStream(stream);
        this.mFile = null;
    }

    public BlockCompressedInputStream(File file) throws IOException {
        this.mFile = new SeekableFileStream(file);
        this.mStream = null;
    }

    public BlockCompressedInputStream(URL url) {
        this.mFile = new SeekableBufferedStream(new SeekableHTTPStream(url));
        this.mStream = null;
    }

    public BlockCompressedInputStream(SeekableStream strm) {
        this.mFile = strm;
        this.mStream = null;
    }

    public void setCheckCrcs(boolean check) {
        this.blockGunzipper.setCheckCrcs(check);
    }

    @Override
    public int available() throws IOException {
        if (this.mCurrentBlock == null || this.mCurrentOffset == this.mCurrentBlock.length) {
            this.readBlock();
        }
        if (this.mCurrentBlock == null) {
            return 0;
        }
        return this.mCurrentBlock.length - this.mCurrentOffset;
    }

    @Override
    public void close() throws IOException {
        if (this.mFile != null) {
            this.mFile.close();
            this.mFile = null;
        } else if (this.mStream != null) {
            this.mStream.close();
            this.mStream = null;
        }
        this.mFileBuffer = null;
        this.mCurrentBlock = null;
    }

    @Override
    public int read() throws IOException {
        return this.available() > 0 ? this.mCurrentBlock[this.mCurrentOffset++] & 0xFF : -1;
    }

    @Override
    public int read(byte[] buffer) throws IOException {
        return this.read(buffer, 0, buffer.length);
    }

    public String readLine() throws IOException {
        int available = this.available();
        if (available == 0) {
            return null;
        }
        if (this.buf == null) {
            this.buf = new ByteArrayOutputStream(8192);
        }
        this.buf.reset();
        boolean done = false;
        boolean foundCr = false;
        while (!done) {
            int linetmpPos = this.mCurrentOffset;
            int bCnt = 0;
            while (available-- > 0) {
                byte c;
                if ((c = this.mCurrentBlock[linetmpPos++]) == 10) {
                    done = true;
                    break;
                }
                if (foundCr) {
                    --linetmpPos;
                    done = true;
                    break;
                }
                if (c == 13) {
                    foundCr = true;
                    continue;
                }
                ++bCnt;
            }
            if (this.mCurrentOffset < linetmpPos) {
                this.buf.write(this.mCurrentBlock, this.mCurrentOffset, bCnt);
                this.mCurrentOffset = linetmpPos;
            }
            if ((available = this.available()) != 0) continue;
            done = true;
        }
        return this.buf.toString();
    }

    @Override
    public int read(byte[] buffer, int offset, int length) throws IOException {
        int originalLength = length;
        while (length > 0) {
            int available = this.available();
            if (available == 0) {
                if (originalLength != length) break;
                return -1;
            }
            int copyLength = Math.min(length, available);
            System.arraycopy(this.mCurrentBlock, this.mCurrentOffset, buffer, offset, copyLength);
            this.mCurrentOffset += copyLength;
            offset += copyLength;
            length -= copyLength;
        }
        return originalLength - length;
    }

    public void seek(long pos) throws IOException {
        int available;
        if (this.mFile == null) {
            throw new IOException("Cannot seek on stream based file");
        }
        long compressedOffset = BlockCompressedFilePointerUtil.getBlockAddress(pos);
        int uncompressedOffset = BlockCompressedFilePointerUtil.getBlockOffset(pos);
        if (this.mBlockAddress == compressedOffset && this.mCurrentBlock != null) {
            available = this.mCurrentBlock.length;
        } else {
            this.mFile.seek(compressedOffset);
            this.mBlockAddress = compressedOffset;
            this.mLastBlockLength = 0;
            this.readBlock();
            available = this.available();
        }
        if (uncompressedOffset > available || uncompressedOffset == available && !this.eof()) {
            throw new IOException("Invalid file pointer: " + pos);
        }
        this.mCurrentOffset = uncompressedOffset;
    }

    private boolean eof() throws IOException {
        if (this.mFile.eof()) {
            return true;
        }
        return this.mFile.length() - (this.mBlockAddress + (long)this.mLastBlockLength) == (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length;
    }

    public long getFilePointer() {
        if (this.mCurrentOffset == this.mCurrentBlock.length) {
            return BlockCompressedFilePointerUtil.makeFilePointer(this.mBlockAddress + (long)this.mLastBlockLength, 0);
        }
        return BlockCompressedFilePointerUtil.makeFilePointer(this.mBlockAddress, this.mCurrentOffset);
    }

    public static long getFileBlock(long bgzfOffset) {
        return BlockCompressedFilePointerUtil.getBlockAddress(bgzfOffset);
    }

    public static boolean isValidFile(InputStream stream) throws IOException {
        if (!stream.markSupported()) {
            throw new RuntimeException("Cannot test non-buffered stream");
        }
        stream.mark(18);
        byte[] buffer = new byte[18];
        int count = BlockCompressedInputStream.readBytes(stream, buffer, 0, 18);
        stream.reset();
        return count == 18 && BlockCompressedInputStream.isValidBlockHeader(buffer);
    }

    private static boolean isValidBlockHeader(byte[] buffer) {
        return buffer[0] == 31 && (buffer[1] & 0xFF) == 139 && (buffer[3] & 4) != 0 && buffer[10] == 6 && buffer[12] == 66 && buffer[13] == 67;
    }

    private void readBlock() throws IOException {
        int count;
        if (this.mFileBuffer == null) {
            this.mFileBuffer = new byte[65536];
        }
        if ((count = this.readBytes(this.mFileBuffer, 0, 18)) == 0) {
            this.mCurrentOffset = 0;
            this.mBlockAddress += (long)this.mLastBlockLength;
            this.mCurrentBlock = new byte[0];
            return;
        }
        if (count != 18) {
            throw new IOException("Premature end of file");
        }
        int blockLength = this.unpackInt16(this.mFileBuffer, 16) + 1;
        if (blockLength < 18 || blockLength > this.mFileBuffer.length) {
            throw new IOException("Unexpected compressed block length: " + blockLength);
        }
        int remaining = blockLength - 18;
        count = this.readBytes(this.mFileBuffer, 18, remaining);
        if (count != remaining) {
            throw new FileTruncatedException("Premature end of file");
        }
        this.inflateBlock(this.mFileBuffer, blockLength);
        this.mCurrentOffset = 0;
        this.mBlockAddress += (long)this.mLastBlockLength;
        this.mLastBlockLength = blockLength;
    }

    private void inflateBlock(byte[] compressedBlock, int compressedLength) throws IOException {
        int uncompressedLength = this.unpackInt32(compressedBlock, compressedLength - 4);
        byte[] buffer = this.mCurrentBlock;
        this.mCurrentBlock = null;
        if (buffer == null || buffer.length != uncompressedLength) {
            try {
                buffer = new byte[uncompressedLength];
            }
            catch (NegativeArraySizeException e) {
                throw new RuntimeException("BGZF file has invalid uncompressedLength: " + uncompressedLength, e);
            }
        }
        this.blockGunzipper.unzipBlock(buffer, compressedBlock, compressedLength);
        this.mCurrentBlock = buffer;
    }

    private int readBytes(byte[] buffer, int offset, int length) throws IOException {
        if (this.mFile != null) {
            return BlockCompressedInputStream.readBytes(this.mFile, buffer, offset, length);
        }
        if (this.mStream != null) {
            return BlockCompressedInputStream.readBytes(this.mStream, buffer, offset, length);
        }
        return 0;
    }

    private static int readBytes(SeekableStream file, byte[] buffer, int offset, int length) throws IOException {
        int bytesRead = 0;
        while (bytesRead < length) {
            int count = file.read(buffer, offset + bytesRead, length - bytesRead);
            if (count <= 0) break;
            bytesRead += count;
        }
        return bytesRead;
    }

    private static int readBytes(InputStream stream, byte[] buffer, int offset, int length) throws IOException {
        int bytesRead = 0;
        while (bytesRead < length) {
            int count = stream.read(buffer, offset + bytesRead, length - bytesRead);
            if (count <= 0) break;
            bytesRead += count;
        }
        return bytesRead;
    }

    private int unpackInt16(byte[] buffer, int offset) {
        return buffer[offset] & 0xFF | (buffer[offset + 1] & 0xFF) << 8;
    }

    private int unpackInt32(byte[] buffer, int offset) {
        return buffer[offset] & 0xFF | (buffer[offset + 1] & 0xFF) << 8 | (buffer[offset + 2] & 0xFF) << 16 | (buffer[offset + 3] & 0xFF) << 24;
    }

    public static FileTermination checkTermination(File file) throws IOException {
        long fileSize = file.length();
        if (fileSize < (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length) {
            return FileTermination.DEFECTIVE;
        }
        RandomAccessFile raFile = new RandomAccessFile(file, "r");
        try {
            raFile.seek(fileSize - (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length);
            byte[] buf = new byte[BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length];
            raFile.readFully(buf);
            if (Arrays.equals(buf, BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK)) {
                FileTermination fileTermination = FileTermination.HAS_TERMINATOR_BLOCK;
                return fileTermination;
            }
            int bufsize = (int)Math.min(fileSize, 65536L);
            buf = new byte[bufsize];
            raFile.seek(fileSize - (long)bufsize);
            raFile.read(buf);
            int i = buf.length - BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length;
            while (i >= 0) {
                if (BlockCompressedInputStream.preambleEqual(BlockCompressedStreamConstants.GZIP_BLOCK_PREAMBLE, buf, i, BlockCompressedStreamConstants.GZIP_BLOCK_PREAMBLE.length)) {
                    ByteBuffer byteBuffer = ByteBuffer.wrap(buf, i + BlockCompressedStreamConstants.GZIP_BLOCK_PREAMBLE.length, 4);
                    byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
                    int totalBlockSizeMinusOne = byteBuffer.getShort() & 0xFFFF;
                    if (buf.length - i == totalBlockSizeMinusOne + 1) {
                        FileTermination fileTermination = FileTermination.HAS_HEALTHY_LAST_BLOCK;
                        return fileTermination;
                    }
                    FileTermination fileTermination = FileTermination.DEFECTIVE;
                    return fileTermination;
                }
                --i;
            }
            FileTermination fileTermination = FileTermination.DEFECTIVE;
            return fileTermination;
        }
        finally {
            raFile.close();
        }
    }

    private static boolean preambleEqual(byte[] preamble, byte[] buf, int startOffset, int length) {
        int i = 0;
        while (i < length) {
            if (preamble[i] != buf[i + startOffset]) {
                return false;
            }
            ++i;
        }
        return true;
    }

    public static enum FileTermination {
        HAS_TERMINATOR_BLOCK,
        HAS_HEALTHY_LAST_BLOCK,
        DEFECTIVE;

    }
}

