/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.illumina;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import net.sf.picard.PicardException;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.cmdline.Usage;
import net.sf.picard.illumina.IlluminaBasecallsToSamConverter;
import net.sf.picard.illumina.parser.ClusterData;
import net.sf.picard.illumina.parser.IlluminaDataProvider;
import net.sf.picard.illumina.parser.IlluminaDataProviderFactory;
import net.sf.picard.illumina.parser.IlluminaDataType;
import net.sf.picard.illumina.parser.ReadStructure;
import net.sf.picard.io.IoUtil;
import net.sf.picard.util.FileChannelJDKBugWorkAround;
import net.sf.picard.util.IlluminaUtil;
import net.sf.picard.util.Log;
import net.sf.picard.util.TabbedTextFileWithHeaderParser;
import net.sf.samtools.BAMRecordCodec;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileWriter;
import net.sf.samtools.SAMFileWriterFactory;
import net.sf.samtools.SAMReadGroupRecord;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMRecordQueryNameComparator;
import net.sf.samtools.util.Iso8601Date;
import net.sf.samtools.util.PeekIterator;
import net.sf.samtools.util.SortingCollection;

public class IlluminaBasecallsToSam
extends CommandLineProgram {
    private static final Log log = Log.getInstance(IlluminaBasecallsToSam.class);
    private static final boolean PRINT_TIMING = false;
    public static final String READ_STRUCTURE_DOC = "A description of the logical structure of clusters in an Illumina Run, i.e. a description of the structure IlluminaBasecallsToSam assumes the  data to be in. It should consist of integer/character pairs describing the number of cycles and the type of those cycles (B for Barcode, T for Template, and S for skip).  E.g. If the input data consists of 80 base clusters and we provide a read structure of \"36T8B8S30T\" then, before being converted to SAM records those bases will be split into 4 reads where read one consists of 36 cycles of template, read two consists of 8 cycles of barcode, read three will be an 8 base read of skipped cycles and read four is another 30 cycle template read.  The read consisting of skipped cycles would NOT be included in output SAM/BAM file read groups.";
    @Usage
    public String USAGE = String.valueOf(this.getStandardUsagePreamble()) + "Generate a SAM or BAM file from data in an Illumina basecalls output directory.\n";
    @Option(doc="The basecalls output directory. ", shortName="B")
    public File BASECALLS_DIR;
    @Option(doc="Lane number. ", shortName="L")
    public Integer LANE;
    @Option(doc="The output SAM or BAM file. Format is determined by extension.", shortName="O", mutex={"BARCODE_PARAMS"})
    public File OUTPUT;
    @Option(doc="Prefixed to read names.")
    public String RUN_BARCODE;
    @Option(doc="The name of the sequenced sample", shortName="ALIAS", mutex={"BARCODE_PARAMS"})
    public String SAMPLE_ALIAS;
    @Option(doc="ID used to link RG header record with RG tag in SAM record.  If these are unique in SAM files that get merged, merge performance is better.  If not specified, READ_GROUP_ID = <first 5 chars of RUN_BARCODE>.<LANE> .", shortName="RG", optional=true)
    public String READ_GROUP_ID;
    @Option(doc="The name of the sequenced library", shortName="LIB", optional=true, mutex={"BARCODE_PARAMS"})
    public String LIBRARY_NAME;
    @Option(doc="The name of the sequencing center that produced the reads to fill in the RG.CN tag.", optional=true)
    public String SEQUENCING_CENTER = "BI";
    @Option(doc="The start date of the run.", optional=true)
    public Date RUN_START_DATE;
    @Option(doc="The name of the sequencing technology that produced the read.", optional=true)
    public String PLATFORM = "illumina";
    @Option(doc="A description of the logical structure of clusters in an Illumina Run, i.e. a description of the structure IlluminaBasecallsToSam assumes the  data to be in. It should consist of integer/character pairs describing the number of cycles and the type of those cycles (B for Barcode, T for Template, and S for skip).  E.g. If the input data consists of 80 base clusters and we provide a read structure of \"36T8B8S30T\" then, before being converted to SAM records those bases will be split into 4 reads where read one consists of 36 cycles of template, read two consists of 8 cycles of barcode, read three will be an 8 base read of skipped cycles and read four is another 30 cycle template read.  The read consisting of skipped cycles would NOT be included in output SAM/BAM file read groups.", shortName="RS")
    public String READ_STRUCTURE;
    @Option(doc="Tab-separated file for creating all output BAMs for barcoded run with single IlluminaBasecallsToSam invocation.  Columns are BARCODE, OUTPUT, SAMPLE_ALIAS, and LIBRARY_NAME.  Row with BARCODE=N is used to specify a file for no barcode match", mutex={"OUTPUT", "SAMPLE_ALIAS", "LIBRARY_NAME"})
    public File BARCODE_PARAMS;
    @Option(doc="Which adapters to look for in the read.")
    public List<IlluminaUtil.IlluminaAdapterPair> ADAPTERS_TO_CHECK = new ArrayList<IlluminaUtil.IlluminaAdapterPair>(Arrays.asList(IlluminaUtil.IlluminaAdapterPair.INDEXED, IlluminaUtil.IlluminaAdapterPair.DUAL_INDEXED, IlluminaUtil.IlluminaAdapterPair.NEXTERA_V2));
    @Option(doc="Run this many TileProcessors in parallel.  If NUM_PROCESSORS = 0, number of cores is automatically set to the number of cores available on the machine. If NUM_PROCESSORS < 0 then the number of cores used will be the number available on the machine less NUM_PROCESSORS.")
    public Integer NUM_PROCESSORS = 0;
    @Option(doc="If set, this is the first tile to be processed (for debugging).  Note that tiles are not processed in numerical order.", optional=true)
    public Integer FIRST_TILE;
    @Option(doc="If set, process no more than this many tiles (for debugging).", optional=true)
    public Integer TILE_LIMIT;
    @Option(doc="If true, call System.gc() periodically.  This is useful in cases in which the -Xmx value passed is larger than the available memory.  Default: enable FORCE_GC if multi-threading.", optional=true)
    public Boolean FORCE_GC;
    @Option(doc="Configure SortingCollections in TileProcessors to store this many records before spilling to disk.  For an indexed run, each SortingCollection gets this value/number of indices.")
    public int MAX_READS_IN_RAM_PER_TILE = 1200000;
    private int recordsWritten = 0;
    private IlluminaBasecallsToSamConverter converter;
    private IlluminaDataProviderFactory factory;
    private ReadStructure readStructure;
    private final Map<String, SAMFileWriter> writersByBarcode = new HashMap<String, SAMFileWriter>();
    public static final IlluminaDataType[] DATA_TYPES_NO_BARCODE = new IlluminaDataType[]{IlluminaDataType.BaseCalls, IlluminaDataType.QualityScores, IlluminaDataType.Position, IlluminaDataType.PF};
    public static final IlluminaDataType[] DATA_TYPES_WITH_BARCODE = Arrays.copyOf(DATA_TYPES_NO_BARCODE, DATA_TYPES_NO_BARCODE.length + 1);

    static {
        IlluminaBasecallsToSam.DATA_TYPES_WITH_BARCODE[IlluminaBasecallsToSam.DATA_TYPES_WITH_BARCODE.length - 1] = IlluminaDataType.Barcodes;
    }

    @Override
    protected int doWork() {
        if (this.OUTPUT != null) {
            IoUtil.assertFileIsWritable(this.OUTPUT);
        }
        if (this.BARCODE_PARAMS != null) {
            IoUtil.assertFileIsReadable(this.BARCODE_PARAMS);
        }
        this.readStructure = new ReadStructure(this.READ_STRUCTURE);
        this.factory = new IlluminaDataProviderFactory(this.BASECALLS_DIR, this.LANE, this.readStructure, IlluminaBasecallsToSam.getDataTypesFromReadStructure(this.readStructure));
        log.info("READ STRUCTURE IS " + this.readStructure.toString());
        List<Integer> tiles = new ArrayList<Integer>(this.factory.getAvailableTiles());
        Collections.sort(tiles, new Comparator<Integer>(){

            @Override
            public int compare(Integer integer1, Integer integer2) {
                String s1 = integer1.toString();
                String s2 = integer2.toString();
                if (s1.length() < s2.length()) {
                    if (s2.startsWith(s1)) {
                        return 1;
                    }
                } else if (s2.length() < s1.length() && s1.startsWith(s2)) {
                    return -1;
                }
                return s1.compareTo(s2);
            }
        });
        if (this.FIRST_TILE != null) {
            int i = 0;
            while (i < tiles.size()) {
                if (((Integer)tiles.get(i)).intValue() == this.FIRST_TILE.intValue()) {
                    tiles = tiles.subList(i, tiles.size());
                    break;
                }
                ++i;
            }
            if (((Integer)tiles.get(0)).intValue() != this.FIRST_TILE.intValue()) {
                throw new PicardException("FIRST_TILE=" + this.FIRST_TILE + ", but that tile was not found.");
            }
        }
        if (this.TILE_LIMIT != null && tiles.size() > this.TILE_LIMIT) {
            tiles = tiles.subList(0, this.TILE_LIMIT);
        }
        if (this.OUTPUT != null) {
            this.writersByBarcode.put(null, this.buildSamFileWriter(this.OUTPUT, this.SAMPLE_ALIAS, this.LIBRARY_NAME, null));
        } else {
            this.populateWritersByBarcode();
        }
        this.converter = new IlluminaBasecallsToSamConverter(this.RUN_BARCODE, this.READ_GROUP_ID, this.factory.getOutputReadStructure(), this.ADAPTERS_TO_CHECK);
        int numProcessors = this.NUM_PROCESSORS == 0 ? Runtime.getRuntime().availableProcessors() : (this.NUM_PROCESSORS < 0 ? Runtime.getRuntime().availableProcessors() + this.NUM_PROCESSORS : this.NUM_PROCESSORS);
        numProcessors = Math.min(numProcessors, tiles.size());
        if (this.FORCE_GC == null) {
            this.FORCE_GC = numProcessors > 1;
        }
        if (numProcessors > 1) {
            FileChannelJDKBugWorkAround.doBugWorkAround();
            log.info("Creating " + numProcessors + " TileProcessors.");
            LinkedList<TileProcessor> activeProcessors = new LinkedList<TileProcessor>();
            Iterator<Integer> tileIterator = tiles.iterator();
            int i = 0;
            while (i < numProcessors) {
                TileProcessor tileProcessor = new TileProcessor();
                tileProcessor.initialize(tileIterator.next());
                tileProcessor.startThread();
                activeProcessors.addLast(tileProcessor);
                ++i;
            }
            while (!activeProcessors.isEmpty()) {
                TileProcessor earliestProcessor = (TileProcessor)activeProcessors.removeFirst();
                this.maybeGC();
                earliestProcessor.waitUntilDone();
                earliestProcessor.sortAndFlush();
                if (!tileIterator.hasNext()) continue;
                earliestProcessor.initialize(tileIterator.next());
                earliestProcessor.startThread();
                activeProcessors.addLast(earliestProcessor);
            }
        } else {
            log.info("Single TileProcessor mode.");
            TileProcessor tileProcessor = new TileProcessor();
            for (int tile : tiles) {
                tileProcessor.initialize(tile);
                tileProcessor.run();
                this.maybeGC();
                tileProcessor.sortAndFlush();
            }
        }
        for (SAMFileWriter writer : this.writersByBarcode.values()) {
            writer.close();
        }
        log.info("Wrote " + this.recordsWritten + " read records total.");
        return 0;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void populateWritersByBarcode() {
        String[] columnLabels;
        TabbedTextFileWithHeaderParser barcodeParamsParser = new TabbedTextFileWithHeaderParser(this.BARCODE_PARAMS);
        String[] stringArray = columnLabels = new String[]{"OUTPUT", "SAMPLE_ALIAS", "LIBRARY_NAME"};
        int n = columnLabels.length;
        int n2 = 0;
        while (n2 < n) {
            String columnLabel = stringArray[n2];
            if (!barcodeParamsParser.hasColumn(columnLabel)) {
                throw new PicardException("BARCODE_PARAMS file " + this.BARCODE_PARAMS + " does not have column " + columnLabel + ".");
            }
            ++n2;
        }
        ArrayList<String> barcodeColumnLabels = new ArrayList<String>();
        if (this.readStructure.barcodes.length() == 1) {
            if (barcodeParamsParser.hasColumn("BARCODE")) {
                barcodeColumnLabels.add("BARCODE");
            } else {
                if (!barcodeParamsParser.hasColumn("BARCODE_1")) throw new PicardException("BARCODE_PARAMS file " + this.BARCODE_PARAMS + " does not have column BARCODE or BARCODE_1.");
                barcodeColumnLabels.add("BARCODE_1");
            }
        } else {
            int i = 1;
            while (i <= this.readStructure.barcodes.length()) {
                String barcodeLabel = "BARCODE_" + i;
                if (!barcodeParamsParser.hasColumn(barcodeLabel)) {
                    throw new PicardException("BARCODE_PARAMS file " + this.BARCODE_PARAMS + " does not have column " + barcodeLabel + ".");
                }
                barcodeColumnLabels.add(barcodeLabel);
                ++i;
            }
        }
        for (TabbedTextFileWithHeaderParser.Row row : barcodeParamsParser) {
            String key;
            ArrayList<String> barcodeValues = new ArrayList<String>();
            for (String barcodeLabel : barcodeColumnLabels) {
                barcodeValues.add(row.getField(barcodeLabel));
            }
            String[] barcodeValuesArray = barcodeValues.toArray(new String[0]);
            String string = key = barcodeValues.contains("N") ? null : IlluminaUtil.barcodeSeqsToString(barcodeValuesArray);
            if (this.writersByBarcode.containsKey(key)) {
                throw new PicardException("Row for barcode " + key + " appears more than once in BARCODE_PARAMS file " + this.BARCODE_PARAMS);
            }
            SAMFileWriter writer = this.buildSamFileWriter(new File(row.getField("OUTPUT")), row.getField("SAMPLE_ALIAS"), row.getField("LIBRARY_NAME"), barcodeValuesArray);
            this.writersByBarcode.put(key, writer);
        }
        if (!this.writersByBarcode.isEmpty()) return;
        throw new PicardException("BARCODE_PARAMS file " + this.BARCODE_PARAMS + " does have any data rows.");
    }

    private void maybeGC() {
        if (this.FORCE_GC.booleanValue()) {
            System.out.println("Before explicit GC, Runtime.totalMemory()=" + Runtime.getRuntime().totalMemory());
            System.gc();
            System.runFinalization();
            System.out.println("After explicit GC, Runtime.totalMemory()=" + Runtime.getRuntime().totalMemory());
        }
    }

    private synchronized void incrementRecordsWritten() {
        ++this.recordsWritten;
        if (this.recordsWritten % 1000000 == 0) {
            log.info(String.valueOf(this.recordsWritten) + " read records written...");
        }
    }

    private SAMFileWriter buildSamFileWriter(File output, String sampleAlias, String libraryName, String[] barcodes) {
        IoUtil.assertFileIsWritable(output);
        SAMReadGroupRecord rg = new SAMReadGroupRecord(this.READ_GROUP_ID);
        rg.setSample(sampleAlias);
        String platformUnit = String.valueOf(this.RUN_BARCODE) + "." + this.LANE;
        if (barcodes != null) {
            platformUnit = String.valueOf(platformUnit) + "." + IlluminaUtil.barcodeSeqsToString(barcodes);
        }
        rg.setAttribute("PU", platformUnit);
        if (libraryName != null) {
            rg.setLibrary(libraryName);
        }
        if (this.SEQUENCING_CENTER != null) {
            rg.setAttribute("CN", this.SEQUENCING_CENTER);
        }
        if (this.PLATFORM != null) {
            rg.setAttribute("PL", this.PLATFORM);
        }
        if (this.RUN_START_DATE != null) {
            Iso8601Date date = new Iso8601Date(this.RUN_START_DATE);
            rg.setAttribute("DT", date.toString());
        }
        SAMFileHeader header = new SAMFileHeader();
        header.setSortOrder(SAMFileHeader.SortOrder.queryname);
        header.addReadGroup(rg);
        return new SAMFileWriterFactory().makeSAMOrBAMWriter(header, true, output);
    }

    public static void main(String[] argv) {
        System.exit(new IlluminaBasecallsToSam().instanceMain(argv));
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> messages = new ArrayList<String>();
        this.readStructure = new ReadStructure(this.READ_STRUCTURE);
        if (!this.readStructure.barcodes.isEmpty() && this.BARCODE_PARAMS == null) {
            messages.add("BARCODE_PARAMS is missing.  If READ_STRUCTURE contains a B (barcode) then BARCODE_PARAMS must be provided!");
        }
        if (this.READ_GROUP_ID == null) {
            this.READ_GROUP_ID = String.valueOf(this.RUN_BARCODE.substring(0, 5)) + "." + this.LANE;
        }
        if (messages.size() == 0) {
            return null;
        }
        return messages.toArray(new String[messages.size()]);
    }

    public static IlluminaDataType[] getDataTypesFromReadStructure(ReadStructure readStructure) {
        if (readStructure.barcodes.isEmpty()) {
            return DATA_TYPES_NO_BARCODE;
        }
        return DATA_TYPES_WITH_BARCODE;
    }

    private static class SamRecordSorter {
        private final Comparator<SAMRecord> comparator = new SAMRecordQueryNameComparator();
        private final SortingCollection.Codec<SAMRecord> codec;
        private final SAMFileWriter writer;
        private final int maxRecordsInRam;
        private final List<File> tmpDirs;
        private SortingCollection<SAMRecord> records;
        private int numRecords = 0;

        private SamRecordSorter(int maxRecordsInRam, SAMFileWriter writer, List<File> tmpDirs) {
            this.maxRecordsInRam = maxRecordsInRam;
            this.tmpDirs = tmpDirs;
            this.codec = new BAMRecordCodec(writer.getFileHeader());
            this.writer = writer;
            this.createSortingCollection();
        }

        private void createSortingCollection() {
            if (this.records != null) {
                this.records.cleanup();
            }
            this.records = SortingCollection.newInstance(SAMRecord.class, this.codec, this.comparator, this.maxRecordsInRam, this.tmpDirs);
        }

        void addAlignment(SAMRecord rec) {
            this.records.add(rec);
            ++this.numRecords;
        }

        void sortAndFlush() {
            this.records.doneAdding();
            PeekIterator it = new PeekIterator(this.records.iterator());
            while (it.hasNext()) {
                SAMRecord rec = (SAMRecord)it.next();
                if (it.hasNext()) {
                    SAMRecord lookAhead = (SAMRecord)it.peek();
                    if (!rec.getReadUnmappedFlag() || !lookAhead.getReadUnmappedFlag()) {
                        throw new IllegalStateException("Should not have mapped reads.");
                    }
                    if (this.comparator.compare(rec, lookAhead) == 0) {
                        it.next();
                        log.info("Skipping reads with identical read names: " + rec.getReadName());
                        continue;
                    }
                }
                this.writer.addAlignment(rec);
            }
            this.createSortingCollection();
            this.numRecords = 0;
        }

        int size() {
            return this.numRecords;
        }

        SAMFileWriter getWriter() {
            return this.writer;
        }
    }

    private class TileProcessor
    implements Runnable {
        private final Map<String, SamRecordSorter> sorters = new HashMap<String, SamRecordSorter>();
        private final Object tileProcessedMonitor = new Object();
        private boolean tileProcessed;
        private int tile;
        private Thread thread;
        private boolean threadFailed;
        private SAMRecord[] records;

        private TileProcessor() {
            int maxRecordsInRam = IlluminaBasecallsToSam.this.MAX_READS_IN_RAM_PER_TILE / IlluminaBasecallsToSam.this.writersByBarcode.size();
            for (Map.Entry entry : IlluminaBasecallsToSam.this.writersByBarcode.entrySet()) {
                this.sorters.put((String)entry.getKey(), new SamRecordSorter(maxRecordsInRam, (SAMFileWriter)entry.getValue(), IlluminaBasecallsToSam.this.TMP_DIR));
            }
            this.records = new SAMRecord[IlluminaBasecallsToSam.this.converter.getNumRecordsPerCluster()];
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void initialize(int tile) {
            this.tile = tile;
            this.thread = null;
            this.threadFailed = false;
            Object object = this.tileProcessedMonitor;
            synchronized (object) {
                this.tileProcessed = false;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void waitUntilDone() {
            Object object = this.tileProcessedMonitor;
            synchronized (object) {
                if (this.threadFailed) {
                    throw new PicardException("TileProcessor thread terminated with an exception");
                }
                if (this.tileProcessed) {
                    return;
                }
                try {
                    this.tileProcessedMonitor.wait();
                    if (this.threadFailed) {
                        throw new PicardException("TileProcessor thread terminated with an exception");
                    }
                }
                catch (InterruptedException e) {
                    throw new PicardException("Waiting for tile to finish.", e);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            block15: {
                try {
                    try {
                        List<Integer> oneTile = Arrays.asList(this.tile);
                        IlluminaDataProvider dataProvider = IlluminaBasecallsToSam.this.factory.makeDataProvider(oneTile);
                        this.processTile(dataProvider);
                    }
                    catch (Throwable e) {
                        if (this.thread == null) {
                            throw new PicardException("Exception in TileProcessor", e);
                        }
                        this.threadFailed = true;
                        log.error(e, "Exception in TileProcessor");
                        Object object = this.tileProcessedMonitor;
                        synchronized (object) {
                            this.tileProcessed = true;
                            this.tileProcessedMonitor.notifyAll();
                            break block15;
                        }
                    }
                }
                catch (Throwable throwable) {
                    Object object = this.tileProcessedMonitor;
                    synchronized (object) {
                        this.tileProcessed = true;
                        this.tileProcessedMonitor.notifyAll();
                    }
                    throw throwable;
                }
                Object object = this.tileProcessedMonitor;
                synchronized (object) {
                    this.tileProcessed = true;
                    this.tileProcessedMonitor.notifyAll();
                }
            }
        }

        public void startThread() {
            this.thread = new Thread(this);
            this.thread.start();
        }

        private void processTile(IlluminaDataProvider dataProvider) {
            Object sqCallWatch = null;
            Object readWatch = null;
            while (dataProvider.hasNext()) {
                ClusterData cluster = dataProvider.next();
                SamRecordSorter sorter = this.sorters.get(cluster.getMatchedBarcode());
                if (sorter == null) {
                    throw new PicardException("Barcode encountered in that was not specified in BARCODE_PARAMS: " + cluster.getMatchedBarcode());
                }
                if (((IlluminaBasecallsToSam)IlluminaBasecallsToSam.this).readStructure.templates.length() != 1 && ((IlluminaBasecallsToSam)IlluminaBasecallsToSam.this).readStructure.templates.length() != 2) {
                    throw new PicardException("Number of templates(" + ((IlluminaBasecallsToSam)IlluminaBasecallsToSam.this).readStructure.templates.length() + ") specified by read structure was greater than 2");
                }
                SAMFileHeader header = sorter.getWriter().getFileHeader();
                IlluminaBasecallsToSam.this.converter.createSamRecords(cluster, header, this.records);
                SAMRecord[] sAMRecordArray = this.records;
                int n = this.records.length;
                int n2 = 0;
                while (n2 < n) {
                    SAMRecord sam = sAMRecordArray[n2];
                    sorter.addAlignment(sam);
                    IlluminaBasecallsToSam.this.incrementRecordsWritten();
                    ++n2;
                }
            }
        }

        void sortAndFlush() {
            for (Map.Entry<String, SamRecordSorter> entry : this.sorters.entrySet()) {
                log.info("Writing " + entry.getValue().size() + " records for tile " + this.tile + " and barcode " + entry.getKey());
                entry.getValue().sortAndFlush();
            }
        }
    }
}

