# ===========================================================================
# CreateGTestTarget.cmake
# -----------------------
#
# This module creates a CppUtilities sub-target for linking the GoogleTest
# library. It exploits the ExternalProject CMake facility for downloading
# GoogleTest from GitHub, checking out release 1.8.1 and installing it
# in the directory specified as input.
# ------------------------------------------------------------------------
#
# function CreateGTestTarget
#   [input] TEST_DEPS_INSTALL_PREFIX : path of the directory GoogleTest is
#                                      installed to.
#   [target] CppUtilities::GTest : name of the created target for linking
#                                  GoogleTest.
# ------------------------------------------------------------------------
#
# author : Emanuele Parisi
# ===========================================================================

function(CreateGTestTarget
         TEST_DEPS_INSTALL_PREFIX)

	include(ExternalProject)

	# Check the installation prefix has been defined.
	if (DEFINED ${TEST_DEPS_INSTALL_PREFIX})
		message(FATAL_ERROR "Missing parameter 'TEST_DEPS_INSTALL_PREFIX'")
	endif()

	# Specify the include/ and lib/ install directory paths.
	set(TEST_DEPS_INSTALL_LIBDIR
	    ${TEST_DEPS_INSTALL_PREFIX}/lib)
	set(TEST_DEPS_INSTALL_INCLUDEDIR
	    ${TEST_DEPS_INSTALL_PREFIX}/include)
	list(APPEND
	     TEST_DEPS_CMAKE_ARGS
	     -DCMAKE_CXX_COMPILER=${CMAKE_CXX_COMPILER}
	     -DCMAKE_BUILD_TYPE=${CMAKE_BUILD_TYPE}
	     -DCMAKE_INSTALL_PREFIX=${TEST_DEPS_INSTALL_PREFIX}
	     -DCMAKE_INSTALL_LIBDIR=${TEST_DEPS_INSTALL_LIBDIR}
	     -DCMAKE_INSTALL_INCLUDEDIR=${TEST_DEPS_INSTALL_INCLUDEDIR})

	# Create the GoogleTest download target.
	ExternalProject_Add(cpputilities_dependencies_googletest
	                    GIT_REPOSITORY
	                    https://github.com/google/googletest.git
	                    GIT_TAG
	                    release-1.8.1
	                    CMAKE_ARGS
	                    ${TEST_DEPS_CMAKE_ARGS}
	                    -DBUILD_GMOCK=0
	                    UPDATE_DISCONNECTED
	                    YES)

	# Guess the GoogleTest library name depending on the CMAKE_BUILD_TYPE.
	if (CMAKE_BUILD_TYPE STREQUAL Debug)
		string(CONCAT
		       TEST_DEPS_GTEST_MAIN_LIBPATH
		       ${TEST_DEPS_INSTALL_LIBDIR}/
		       ${CMAKE_STATIC_LIBRARY_PREFIX}
		       gtest_maind
		       ${CMAKE_STATIC_LIBRARY_SUFFIX})
		string(CONCAT
		       TEST_DEPS_GTEST_LIBPATH
		       ${TEST_DEPS_INSTALL_LIBDIR}/
		       ${CMAKE_STATIC_LIBRARY_PREFIX}
		       gtestd
		       ${CMAKE_STATIC_LIBRARY_SUFFIX})
	else ()
		string(CONCAT
		       TEST_DEPS_GTEST_MAIN_LIBPATH
		       ${TEST_DEPS_INSTALL_LIBDIR}/
		       ${CMAKE_STATIC_LIBRARY_PREFIX}
		       gtest_main
		       ${CMAKE_STATIC_LIBRARY_SUFFIX})
		string(CONCAT
		       TEST_DEPS_GTEST_LIBPATH
		       ${TEST_DEPS_INSTALL_LIBDIR}/
		       ${CMAKE_STATIC_LIBRARY_PREFIX}
		       gtest
		       ${CMAKE_STATIC_LIBRARY_SUFFIX})
	endif()

	# As ExternalProject actually download and install the library at build
	# time, make sure that an empty include directory exists at configure time.
	file(MAKE_DIRECTORY
	     ${TEST_DEPS_INSTALL_INCLUDEDIR})

	# Create the GoogleTest Main library target.
	add_library(CppUtilities_GTest_Main
	            STATIC IMPORTED)
	add_dependencies(CppUtilities_GTest_Main
	                 cpputilities_dependencies_googletest)
	set_target_properties(CppUtilities_GTest_Main
	                      PROPERTIES
	                      IMPORTED_LOCATION
	                      ${TEST_DEPS_GTEST_MAIN_LIBPATH})

	# Create the GoogleTest library target.
	add_library(CppUtilities_GTest_GTest
	            STATIC IMPORTED)
	add_dependencies(CppUtilities_GTest_GTest
	                 CppUtilities_GTest_Main)
	set_target_properties(CppUtilities_GTest_GTest
	                      PROPERTIES
	                      IMPORTED_LOCATION
	                      ${TEST_DEPS_GTEST_LIBPATH})

	# Create an alias for wrapping both the library in a single target.
	add_library(CppUtilities_GTest
	            INTERFACE)
	target_include_directories(CppUtilities_GTest
	                           INTERFACE
	                           ${TEST_DEPS_INSTALL_INCLUDEDIR})
	target_link_libraries(CppUtilities_GTest
	                      INTERFACE
	                      CppUtilities_GTest_GTest
	                      CppUtilities_GTest_Main
	                      pthread)
	add_library(CppUtilities::GTest
	            ALIAS
	            CppUtilities_GTest)
endfunction()
