/**
 * \file      stats/resource_usage.h
 * \author    Emanuele Parisi
 * \date      March, 2019
 * \copyright Copyright 2019 Emanuele Parisi. All rights reserved.
 *
 * Contains a class for basic in-source resource usage measurements.
 *
 * In this file are contained facilities for performing basic in-source
 * resource usage measurements. This class in an interface for the facilities
 * offered by the POSIX struct rusage data structure; further details about
 * the services offered by this struct can be found in the linux man pages.
 *
 * \see http://man7.org/linux/man-pages/man2/getrusage.2.html
 */

#ifndef CPPUTILS_STATS_RESOURCE_USAGE_H_
#define CPPUTILS_STATS_RESOURCE_USAGE_H_

#include <sys/time.h>
#include <sys/resource.h>

namespace cpputils
{

/**
 * Class for retrieving resources usage statistics for the current process.
 */
class ResourceUsage
{
public:

	/**
	 * Class constructor.
	 *
	 * \param who is a flag for specifying what is the subject the reported
	 * statistics refer to. The default value for this field is RUSAGE_SELF.
	 */
	ResourceUsage (int who = RUSAGE_SELF)
	{
		auto result = getrusage(who,
		                        &usage_);

		if (result == EFAULT)
		{
			std::runtime_error("The usage struct pointer points outside the "
			                   "accessible address space.");
		}
		if (result == EINVAL)
		{
			std::runtime_error("The rusage flag is not valid.");
		}
		who_ = who;
	}

	/**
	 * Class copy constructor.
	 *
	 * \param other is object the current instance is built from.
	 */
	ResourceUsage (const ResourceUsage& other) noexcept
	{
		usage_ = other.usage_;
	}

	/**
	 * Class copy assignment operator.
	 *
	 * \param other is the object the current instance is initialized from.
	 * \return a reference to the initialize object.
	 */
	ResourceUsage&
	operator= (const ResourceUsage& other) noexcept
	{
		if (this != &other)
		{
			usage_ = other.usage_;
		}

		return *this;
	}

	/**
	 * Retrieve the user CPU time used of the current process.
	 *
	 * Return the total amount of time spent executing in user mode, expressed
	 * in microseconds.
	 *
	 * \return the user CPU time of the current process expressed in
	 * microseconds.
	 */
	inline auto
	getUserCPUTime () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_utime.tv_sec * 1000000 + usage_.ru_utime.tv_usec;
	}

	/**
	 * Retrieve the system CPU time used of the current process.
	 *
	 * Returns the total amount of time spent executing in kernel mode,
	 * expressed in microseconds.
	 *
	 * \return the system CPU time of the current process expressed in
	 * microseconds.
	 */
	inline auto
	getSystemCPUTime () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_stime.tv_sec * 1000000 + usage_.ru_stime.tv_usec;
	}

	/**
	 * Retrieve the maximum resident set size of the current process.
	 *
	 * Return the maximum resident set size used expressed in kilobytes.
	 * If the class is initialized with the RUSAGE_CHILDREN flag, this is the
	 * resident set size of the largest child, not the maximum resident set
	 * size of the process tree.
	 *
	 * \return the maximum resident set size of the current process expressed
	 * in kilobytes.
	 */
	inline auto
	getMaximumResidentSetSize () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_maxrss;
	}

	/**
	 * Retrieve the number of soft page faults for the current process.
	 *
	 * Return the number of page faults serviced without any I/O activity; here
	 * I/O activity is avoided by “reclaiming” a page frame from the list of
	 * pages awaiting reallocation.
	 *
	 * \return the number of soft page faults for the current process.
	 */
	inline auto
	getSoftPageFaults () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_minflt;
	}

	/**
	 * Retrieve the number of hard page faults for the current process.
	 *
	 * Return the number of page faults serviced that required I/O activity.
	 *
	 * \return the number of hard page faults for the current process.
	 */
	inline auto
	getHardPageFaults () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_majflt;
	}

	/**
	 * Retrieve the number of times the filesystem had to perform input.
	 *
	 * \return the number of time the filesystem performed input operation.
	 */
	inline auto
	getBlockInputOperations () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_inblock;
	}

	/**
	 * Retrieve the number of times the filesystem had to perform output.
	 *
	 * \return the number of time the filesystem performed output operation.
	 */
	inline auto
	getBlockOutputOperations () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_oublock;
	}

	/**
	 * Retrieve the number of voluntary context switches for the current
	 * process.
	 *
	 * Return the number of times a context switch resulted due to a process
	 * voluntarily giving up the processor before its time slice was completed
	 * (usually to await availability of a resource).
	 *
	 * \return the number of voluntary context switches for the current process.
	 */
	inline auto
	getVoluntaryContextSwitches () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_nvcsw;
	}

	/**
	 * Retrieve the number of involuntary context switches for the current
	 * process.
	 *
	 * Return the number of times a context switch resulted due to a higher
	 * priority process becoming runnable or because the current process
	 * exceeded its time slice.
	 *
	 * \return the number of involuntary context switches for the current
	 * process.
	 */
	inline auto
	getInvoluntaryContextSwitches () noexcept
	{
		getrusage(who_,
		          &usage_);

		return usage_.ru_nivcsw;
	}

private:
	/**
	 * Default flag used for fetching the resource usage. Once it is
	 * initialized in the constructor, it can never be changed.
	 */
	int           who_;
	/**
	 * POSIX native data structure for fetching the requested system
	 * information.
	 */
	struct rusage usage_;
};

} // cpputils

#endif // CPPUTILS_STATS_RESOURCE_USAGE_H_
